<?php

/**

 * An abstract / main controller of the module.

 *

 * @author    PrestaMonster.com | support@prestamonster.com

 * @copyright PrestaMonster.com

 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)

 */



if (!defined('_PS_VERSION_')) {

    exit;

}



require_once dirname(__FILE__).'/classes/HsAccessoriesGroupAbstract.php';

require_once dirname(__FILE__).'/classes/HsMaSpecificPrice.php';

require_once dirname(__FILE__).'/classes/HsMaProduct.php';

require_once dirname(__FILE__).'/classes/HsMaDisplayStyle.php';



abstract class HsMultiAccessoriesAbstract extends Module

{

    const DEFAULT_QTY = 1;



    /**

     * The toolkit which is responsible for installing.

     *

     * @var object

     */

    protected $installer;



    /**

     * The toolkit which is responsible for uninstalling.

     *

     * @var object

     */

    protected $uninstaller;



    /**

     * Path to js folder.

     *

     * @var string

     */

    const PATH_JS = 'abstract/views/js/';



    /**

     * Path to css folder.

     *

     * @var string

     */

    const PATH_CSS = 'abstract/views/css/';



    /**

     * Path to image folder.

     *

     * @var string

     */

    const PATH_IMG = 'abstract/views/img/';



    /**

     * A list of translatable texts.

     *

     * @var array

     */

    public $i18n = array();



    /**

     * A list of urls to be used.

     *

     * @var array

     */

    protected $urls = array();



    /**

     * check if cart is exists or not.

     *

     * @var bool

     */

    protected static $has_cart_been_added = false;



    /**

     * Array configuration key using setting.

     *

     * @var array

     *            <pre>

     *            array(

     *            'HSMA_DISPLAY_STYLE' => 'isString',

     *            'HSMA_SHOW_IMAGES' => 'isString',

     *            'HSMA_APPLY_FANCYBOX_TO_IMAGE' => 'isInt',

     *            'HSMA_SHOW_PRICE' => 'isString',

     *            'HSMA_SHOW_PRICE_TABLE' => 'isString',

     *            'HSMA_TITLE' => 'isString',

     *            'HSMA_MESSAGE_AVAILABLE_LATER' => 'isString',

     *            'HSMA_BUY_ACCESSORY_MAIN_TOGETHER' => 'isInt',

     *            'HSMA_ALERT_MESSAGE' => 'isString',

     *            'HSMA_SHOW_CUSTOM_QUANTITY' => 'isInt',

     *            'HSMA_CHANGE_MAIN_PRICE' => 'isInt'

     *            )

     */

    public $configuration_keys = array(

        'HSMA_DISPLAY_STYLE' => 'isInt',

        'HSMA_SHOW_IMAGES' => 'isInt',

        'HSMA_SHOW_SHORT_DESCRIPTION' => 'isInt',

        'HSMA_SHOW_PRICE' => 'isInt',

        'HSMA_SHOW_PRICE_TABLE' => 'isInt',

        'HSMA_TITLE' => 'isString',

        'HSMA_MESSAGE_AVAILABLE_LATER' => 'isString',

        'HSMA_EACH_ACCESSORY_TO_BASKET' => 'isInt',

        'HSMA_OPEN_ACCESSORIES_IN_NEW_TAB' => 'isInt',

        'HSMA_BUY_ACCESSORY_MAIN_TOGETHER' => 'isInt',

        'HSMA_ALERT_MESSAGE' => 'isString',

        'HSMA_SHOW_CUSTOM_QUANTITY' => 'isInt',

        'HSMA_CHANGE_MAIN_PRICE' => 'isInt',

        'HSMA_APPLY_FANCYBOX_TO_IMAGE' => 'isInt',

        'HSMA_IMAGE_SIZE_IN_FANCYBOX' => 'isString',

    );



    /**

     * Array configuration language keys using setting.

     *

     * @var array

     *            <pre>

     *            array(

     *            'HSMA_TITLE',

     *            'HSMA_MESSAGE_AVAILABLE_LATER',

     *            'HSMA_ALERT_MESSAGE',

     *            )

     */

    public $configuration_lang_keys = array(

        'HSMA_TITLE',

        'HSMA_MESSAGE_AVAILABLE_LATER',

        'HSMA_ALERT_MESSAGE',

    );



    /**

     * check if hookActionCartSave is executed?

     *

     * @var bool

     */

    protected static $is_executed = false;



    /**

     * a list of id accessory.

     *

     * @var array

     */

    protected $accessories = array();



    /**

     * construct function.

     */

    public function __construct()

    {

        $this->tab = 'front_office_features';

        $this->author = 'PrestaMonster.com';

        parent::__construct();

        if (defined('_PS_ADMIN_DIR_')) {

            $this->assignAdminUrls();

        }

        $this->initTranslations();

    }



    /**

     * Install module.

     *

     * @return bool

     */

    public function install()

    {

        $success = array();

        $success[] = parent::install();



        $success[] = $this->installer->installTables();

        $success[] = $this->installer->addNewImageType();



        $success[] = $this->installer->installTabs();

        $success[] = $this->installer->installTab243($this->tab_admin_welcome_page);

        $success[] = $this->installer->installTab27($this->tabs27);



        $success[] = $this->installer->installConfigs();

        $success[] = $this->setDefaultAlertMessage();

        $success[] = $this->setDefaultTitle();



        $success[] = $this->registerHooks();



        return array_sum($success) >= count($success);

    }



    /**

     * Register neccessary hooks.

     *

     * @return bool

     */

    protected function registerHooks()

    {

        $success = array();

        // display at Back office

        $success[] = $this->registerHook('displayBackOfficeHeader');

        $success[] = $this->registerHook('displayAdminListBefore');

        $success[] = $this->registerHook('displayAdminProductsExtra');



        // action at Back office

        $success[] = $this->registerHook('actionProductDelete');

        $success[] = $this->registerHook('actionProductAdd');



        // display at Front office

        $success[] = $this->registerHook('displayHeader');

        $success[] = $this->registerHook('displayFooter');

        $success[] = $this->registerHook('displayRightColumnProduct');

        $success[] = $this->registerHook('displayMultiAccessoriesProduct');

        $success[] = $this->registerHook('displayShoppingCartFooter'); // this is unstable yet



        // action at Front office

        $success[] = $this->registerHook('actionCartSave');

        $success[] = $this->registerHook('actionObjectCartAddAfter');



        return array_sum($success) >= count($success);

    }



    /** Uninstall module

     * @return bool

     */

    public function uninstall()

    {

        return (parent::uninstall() && $this->uninstaller->uninstallTabs() && $this->uninstaller->uninstallTables());

    }



    /**

     * Set default alert message when install module.

     *

     * @return bool

     */

    protected function setDefaultAlertMessage()

    {

        // set default alert message when install

        $flag = true;

        $languages = Language::getLanguages(false);

        $titles = array();

        if (!empty($languages)) {

            foreach ($languages as $language) {

                $titles[$language['id_lang']] = $this->i18n['oops_you_can_not_buy_without_one_of_these_accessories'];

            }

        }

        if (!empty($titles)) {

            $flag = $flag && Configuration::updateValue('HSMA_ALERT_MESSAGE', $titles);

        }



        return $flag;

    }



    /**

     * hook header.

     */

    public function hookDisplayHeader()

    {

        if (!$this->isHookedInCurrentPage()) {

            return;

        }

        

        if ($this->isPrestashop16()) {

            $this->context->controller->addJqueryUI('ui.tooltip');

            $this->context->controller->addCSS($this->getCSSPath().'multiaccessories_16.css', 'all'); 

            $this->context->controller->addJS($this->getJsPath().'accessoriesprice_16.js');

        } else {

            $this->context->controller->addCSS($this->getCSSPath().'multiaccessories_15.css', 'all');

            $this->context->controller->addJS($this->getJsPath().'accessoriesprice_15.js');

        }

        $this->context->controller->addJS($this->getJsPath().'hsma_display_style.js');

        $this->context->controller->addJS($this->getJsPath().'admin_product_setting.js');

        $this->context->controller->addJS($this->getJsPath().'pricetable.js');

        

        $this->context->controller->addJS($this->getJsPath().'multi_accessories.js');



        $this->context->controller->addCSS($this->getCSSPath().'tableprice.css', 'all');

        $this->context->controller->addCSS($this->getCSSPath().'multiaccessories.css', 'all');

        

       

        $this->assignSmartyVariables();



        return $this->display($this->name.'.php', 'display_header.tpl');

    }



    /**

     * hook footer.

     */

    public function hookDisplayFooter()

    {

        if (!$this->isHookedInCurrentPage()) {

            return;

        }

        if ($this->isPrestashop16()) {

            $add_to_basket_js_file = $this->isEnableBlockCartAjax() ? 'accessoriescart_16.js' : 'accessories_add_to_cart.js';

        } else {

            $add_to_basket_js_file = $this->isEnableBlockCartAjax() ? 'accessoriescart_15.js' : 'accessories_add_to_cart.js';

        }

        $this->context->controller->addJS($this->getJsPath().$add_to_basket_js_file);

        $this->context->controller->addJS($this->getJsPath().'pricetable.js');

    }



    /**

     * To decide if we should utilize module blockcart's function or not (when adding an accessory to basket).

     */

    protected function isEnableBlockCartAjax()

    {

        // probably, we only need to consider about that in product page; otherwsie, disable!

        if ($this->context->controller instanceof ProductController) {

            if (Module::isEnabled('blockcart')) {

                if (Configuration::get('PS_BLOCK_CART_AJAX')) {

                    return true;

                }

            }

        }

    }



    /**

     * @return bool

     */

    protected function isProductPage()

    {

        return $this->context->controller instanceof ProductController;

    }



    /**

     * Check if Multi Accessories block is available in current page.

     *

     * @return bool

     */

    protected function isHookedInCurrentPage()

    {

        if ($this->context->controller instanceof ProductController) {

            return true;

        }

        if ($this->context->controller instanceof OrderController || $this->context->controller instanceof OrderOpcController) {

            if ($this->isRegisteredInHook('displayShoppingCart') || $this->isRegisteredInHook('displayShoppingCartFooter')) {

                return true;

            }

        }



        return false;

    }



    /**

     * hook into Back Office header position.

     *

     * @return assign template

     */

    public function hookDisplayBackOfficeHeader()

    {

        if (!($this->context->controller instanceof AdminProductsController) && !($this->context->controller instanceof AdminHsMultiAccessoriesWelcomePageProController) && !($this->context->controller instanceof AdminHsMultiAccessoriesGroupProController)) {

            return;

        }

        /* Include new js */

        $this->context->controller->addJS($this->getJsPath().'admin_product_setting.js');

        $this->context->controller->addJS($this->getJsPath().'admin_multi_accessories.js');

        /* End of include new js */

        $this->context->controller->addCSS($this->getCssPath().'accessory_admin_tab.css');

        if (!$this->isPrestashop16()) {

            $this->context->controller->addCSS($this->getCssPath().'accessory_admin_tab_15.css');

        }

    }



    /**

     * Assign all possible urls which will be in use within the module.

     */

    protected function assignAdminUrls()

    {

        $this->urls = array(

            'ajaxSaveNameUrl' => $this->getTargetUrl($this->class_controller_admin_group, 'saveName'),

            'ajaxAddAccessory' => $this->getTargetUrl($this->class_controller_admin_group, 'addAccessory'),

            'ajaxDeleteAccessory' => $this->getTargetUrl($this->class_controller_admin_group, 'deleteAccessory'),

            'ajaxChangeProductCombination' => $this->getTargetUrl($this->class_controller_admin_group, 'changeProductCombination'),

            'ajaxRegenerateThumbnails' => $this->getTargetUrl($this->class_controller_admin_group, 'regenerateThumbnails'),

            'ajaxChangeDisplayStyle' => $this->getTargetUrl($this->class_controller_admin_group, 'changeDisplayStyle'),

            'ajaxChangeMinQuantity' => $this->getTargetUrl($this->class_controller_admin_group, 'changeMinQuantity'),

            'ajaxChangeDefaultQuantity' => $this->getTargetUrl($this->class_controller_admin_group, 'changeDefaultQuantity'),

            'ajaxAutoCompleteSearch' => $this->getTargetUrl($this->class_controller_accessory_search, 'autoCompleteSearch'),

            'ajaxChangeProductSettingBuyTogether' => $this->getTargetUrl($this->class_controller_admin_product_setting, 'changeProductSettingBuyTogether'),

            'ajaxChangeCustomDisplayedName' => $this->getTargetUrl($this->class_controller_admin_product_setting, 'changeCustomDisplayedName'),

            'ajaxChangeAccessorySettingBuyTogether' => $this->getTargetUrl($this->class_controller_admin_group, 'changeAccessorySettingBuyTogether'),

            'ajaxProcessUpdateAccessoryProductPosition' => $this->getTargetUrl($this->class_controller_admin_group, 'updateAccessoryProductPosition'),

        );

    }



    /**

     * All translatable texts which can be easy to use in Smarty or any module controllers.

     *

     * <br />

     * For example:<br />

     * - Smarty<br />

     * {$hs_i18n.text_1}<br />

     * - controller<br />

     * $this->module->i18n[text_1]

     */

    protected function initTranslations()

    {

        $group_accessory_link = '';

        if (($this->context->controller instanceof AdminProductsController) || ($this->context->controller instanceof AdminHsMultiAccessoriesWelcomePageProController) || ($this->context->controller instanceof AdminHsMultiAccessoriesGroupProController)) {

            $group_accessory_link = $this->context->link->getAdminLink($this->class_controller_admin_group);

        }

        $source = basename(__FILE__, '.php');

        $this->i18n = array(

            'add_to_cart' => $this->l('Add to cart', $source),

            'an_error_occurred_while_attempting_to_move_this_accessory' => $this->l('An error occurred while attempting to move this accessory.', $source),

            'select_accessory' => $this->l('Select an accessory', $source),

            'there_is_not_any_accessory_group' => sprintf($this->l("There is not any accessory group. Let's %s create the first one. %s", $source), '<a href="'.$group_accessory_link.'">', '</a>'),

            'id' => $this->l('ID', $source),

            'group_name' => $this->l('Group name', $source),

            'active' => $this->l('Active', $source),

            'multi_accessories' => $this->l('Multi Accessories', $source),

            'delete_selected_items' => $this->l('Delete selected items', $source),

            'checkbox' => $this->l('Checkbox', $source),

            'dropdown' => $this->l('Dropdown', $source),

            'radio' => $this->l('Radio', $source),

            'settings' => $this->l('Settings', $source),

            'display_style' => $this->l('Display style', $source),

            'define_how_accessories_look_like_at_product_page' => $this->l('Define how accessories look like at product page.', $source),

            'display_images_along_with_each_accessory' => $this->l('Display images along with each accessory.', $source),

            'display_price_along_with_each_accessory' => $this->l('Display price along with each accessory.', $source),

            'tell_your_customers_a_summary' => $this->l('Tell your customers a summary of which accessories to pick up and how much to pay.', $source),

            'add_an_icon_where_people_can_read_description_instead_of_open_that_accessory' => $this->l('Add an icon where people can read description instead of open that accessory.', $source),

            'title_of_accessory_block_at_product_page' => $this->l('Title of accessory block at product page.', $source),

            'show_images' => $this->l('Show images', $source),

            'show_price' => $this->l('Show price', $source),

            'show_short_description' => $this->l('Show short description', $source),

            'show_price_table' => $this->l('Show price table', $source),

            'title' => $this->l('Title', $source),

            'save' => $this->l('Save', $source),

            'open_new_tab' => $this->l('Open in a new tab', $source),

            'view' => $this->l('view', $source),

            'must_have_accessories' => $this->l('Must-have accessories', $source),

            'save_and_stay' => $this->l('Save and stay', $source),

            'cancel' => $this->l('Cancel', $source),

            'sub_total' => $this->l('Sub total', $source),

            'oops_you_can_not_buy_without_one_of_these_accessories' => $this->l('Oops, you can not buy without one of these accessories!', $source),

            'quantity' => $this->l('Quantity', $source),

            'edit_group' => $this->l('Edit group', $source),

            'add_a_new_accessory_group' => $this->l('Add a new accessory group', $source),

            'default_quantity' => $this->l('Default quantity', $source),

            'group' => $this->l('Group', $source),

            'name' => $this->l('Name', $source),

            'active' => $this->l('Active:', $source),

            'enabled' => $this->l('Enabled', $source),

            'disabled' => $this->l('Disabled', $source),

            'ok' => $this->l('ok', $source),

            'error' => $this->l('error', $source),

            'search_for_items' => $this->l('Search for items ...', $source),

            'accessory_group' => $this->l('Accessory group', $source),

            'invalid_characters' => $this->l('Invalid characters:', $source),

            'alert_message' => $this->l('Alert message', $source),

            'tell_your_customer_when_they_dont_choose_any_accessories_to_buy_together_with_main_product' => $this->l('Tell your customer when they don\'t choose any accessories to buy together with main product.', $source),

            'apply_fancybox_to_images' => $this->l('Apply Fancybox to images', $source),

            'show_accessory_images_in_a_fancybox' => $this->l('Show accessory images in a Fancybox.', $source),

            'image_size_in_fancybox' => $this->l('Image size in Fancybox', $source),

            'display_prices_along_with_each_accessory' => $this->l('Display prices along with each accessory.', $source),

            'change_the_main_item_s_price_accordingly' => $this->l('Change the main item\'s price accordingly', $source),

            'whenever_an_accessory_is_added_or_removed_the_main_item_s_price_is_changed_and_your_customers_clearly_know_the_amount' => $this->l('Whenever an accessory is added or removed, the main item\'s price is changed, and your customers clearly know the amount.', $source),

            'add_custom_quantity_to_basket' => $this->l('Add custom quantity to basket', $source),

            'allow_customer_add_custom_quantity_of_each_accessory_to_basket' => $this->l('Allow customer add custom quantity of each accessory to basket.', $source),

            'buy_main_product_accessories_together' => $this->l('Buy main product & accessories together', $source),

            'tell_your_customers_that_they_need_to_buy_main_product_and_accessories_together' => $this->l('Tell your customers that they need to buy main product and accessories together.', $source),

            'add_each_accessory_to_basket' => $this->l('Add each accessory to basket', $source),

            'allow_customer_add_separated_accessory_to_basket' => $this->l('Allow customer add separated accessory to basket.', $source),

            'open_accessories_in_a_new_tab' => $this->l('Open accessories in a new tab', $source),

            'global_update' => $this->l('Global update', $source),

            'select_a_combination_optional' => $this->l('Select a combination (optional)', $source),

            'click_to_view_details' => $this->l('Click to view details', $source),

            'you_must_save_this_product_before_adding_accessories' => $this->l('You must save this product before adding accessories', $source),

            'update_successful' => $this->l('Update successful', $source),

            'use_default' => $this->l('Use default', $source),

            'accessory_is_out_of_stock' => $this->l('Oops! This item is out of stock.', $source),

            'there_is_not_enough_product_in_stock' => $this->l('There is not enough product in stock.', $source),

            'yes' => $this->l('Yes', $source),

            'you_do_not_have_the_right_permission' => $this->l('You do not have the right permission', $source),

            'no' => $this->l('No', $source),

            'use_default' => $this->l('Use default', $source),

            'let_me_specify' => $this->l('Let me specify', $source),

            'buy_main_product_accessory_together' => $this->l('Buy main product accessory together', $source),

            'product_settings' => $this->l('Product settings', $source),

            'required' => $this->l('Required?', $source),

            'if_the_text_displayed_text_when_backordering_is_allowed_in_product_edit_page_is_empty' => $this->l('If the text "Displayed text when backordering is allowed" in product edit page is empty , this message will be displayed.', $source),

            'displayed_text_when_backordering_is_allowed' => $this->l('Displayed text when backordering is allowed', $source),

            'out_of_stock_but_backordering_is_allowed' => $this->l('Out of stock but backordering is allowed.', $source),

            'out_of_stock' => $this->l('Out of stock', $source),

            'only_use_custom_displayed_names_for_this_product' => $this->l('Only use custom displayed names for this product', $source),

            'otherwise_wherever_that_accessory_is_displayed' => $this->l('Otherwise, wherever that accessory is displayed (in Multi Accessories block only), they share the same displayed name.', $source),

            'advanced_settings_for_this_product_only' => $this->l('Advanced settings (for this product only)', $source),

            'accessory' => $this->l('Accessory', $source),

            'displayed_name' => $this->l('Displayed name', $source),

            'price' => $this->l('Price', $source),

            'minimum_quantity' => $this->l('Minimum quantity', $source),

            'position' => $this->l('Position', $source),

            'action' => $this->l('Action', $source),

            'item_inside' => $this->l('%s item inside', $source),

            'items_inside' => $this->l('%s items inside', $source),

            'click_to_edit' => $this->l('Click to edit', $source),

            'there_is_no_accessory_in_this_group' => $this->l('There is no accessory in this group.', $source),

        );

        $this->context->smarty->assign('hs_i18n', $this->i18n);

    }



    /**

     * combine an Ajax URL for the default controller of module.

     *

     * @param string $action

     *

     * @return string full Ajax Url

     */

    public function getTargetUrl($controller = '', $action = '', $ajax = true)

    {

        $params = array();

        $params['ajax'] = $ajax;

        $params['controller'] = $controller;

        $action = trim($action);

        if (!empty($action)) {

            $params['action'] = $action;

        }

        $params['token'] = Tools::getAdminTokenLite($controller);

        $query = array();

        foreach ($params as $key => $value) {

            $query[] = $key.'='.$value;

        }



        return '?'.implode('&', $query);

    }



    /**

     * Get relative path to js files of module.

     *

     * @return string

     */

    public function getJsPath()

    {

        return $this->_path.self::PATH_JS;

    }



    /**

     * Get relative path to css files of module.

     *

     * @return string

     */

    public function getCssPath()

    {

        return $this->_path.self::PATH_CSS;

    }



    /**

     * Get relative path to images files of module.

     *

     * @return string

     */

    public function getImgPath()

    {

        return $this->_path.self::PATH_IMG;

    }



    /**

     * Form Config Methods.

     */

    public function getContent()

    {

        Tools::redirectAdmin($this->context->link->getAdminLink($this->class_controller_admin_group));

    }



    protected function assignSmartyVariables()

    {

        $id_product = $this->isProductPage() ? $this->context->controller->getProduct()->id : 0;

        $link = new Link();

        $this->context->smarty->assign(array(

            'accessory_alert_message' => Configuration::get('HSMA_ALERT_MESSAGE', $this->context->language->id),

            'msg_accessory_is_out_of_stock' => $this->i18n['accessory_is_out_of_stock'],

            'utilize_block_cart_ajax' => (int) $this->isEnableBlockCartAjax(),

            'order_url' => Configuration::get('PS_ORDER_PROCESS_TYPE') == 1 ? $link->getPageLink('order-opc') : $link->getPageLink('order'),

            'buy_main_accessory_together' => $id_product ? HsMaProductSetting::getBuyTogetherCurrentValue($id_product) : 0,

            'display_style' => Configuration::get('HSMA_DISPLAY_STYLE'),

        ));

    }

    /**

     * HOOK_SHOPPING_CART_EXTRA.

     *

     * @param array $params see self::hookDisplayShoppingCartFooter()

     */

    public function hookDisplayShoppingCart(array $params = array())

    {

        return $this->hookDisplayShoppingCartFooter($params);

    }



    /**

     * HOOK_SHOPPING_CART.

     *

     * @param array $params output of Cart::getSummaryDetails()

     *                      <pre>

     *                      array(

     *                      'delivery' => Address,

     *                      'delivery_state' => ?,

     *                      'invoice' => Address,

     *                      'invoice_state' => ?,

     *                      'formattedAddresses' => array,

     *                      'products' => array,

     *                      'gift_products' => array,

     *                      'discounts' => array,

     *                      'is_virtual_cart' => int (0/1)

     *                      'total_discounts' => float,

     *                      'total_discounts_tax_exc' => float,

     *                      'total_wrapping' => float,

     *                      'total_wrapping_tax_exc' => float,

     *                      'total_shipping' => float,

     *                      'total_shipping_tax_exc' => float,

     *                      'total_products_wt' => float,

     *                      'total_products' => float,

     *                      'total_price' => float,

     *                      'total_tax' => float,

     *                      'total_price_without_tax' => float,

     *                      'is_multi_address_delivery' => ?,

     *                      'free_ship' => ?,

     *                      'carrier' => Carrier,

     *                      'cookie' => Cookie,

     *                      'cart' => Cart

     *                      )

     */

    public function hookDisplayShoppingCartFooter(array $params = array())

    {

        return;

        $id_products = $this->getIdProductsFromCart();

        if (empty($id_products)) {

            return;

        }

        $this->context->controller->addJS(_THEME_JS_DIR_.'product.js');

        if (Tools::getValue('ajax')) {

            $this->assignSmartyVariables();

        }



        return $this->renderBlockAccessories($id_products, false);

    }



    /**

     * @return array

     *               <pre>

     *               array(

     *               int,

     *               int,

     *               ...

     *               )

     */

    protected function getIdProductsFromCart()

    {

        $products = $this->context->cart->getProducts();

        $id_products = array();

        foreach ($products as $product) {

            $id_products[] = $product['id_product'];

        }



        return $id_products;

    }



    /**

     * Get table prices of main product + accessories products.

     *

     * @param array $accessory

     *                         <pre>

     *                         array(

     *                         'id_accessory_group' => int,

     *                         'id_accessory' => int,

     *                         'default_quantity' => int,

     *                         'required' => boolean,

     *                         'position' => int,

     *                         'link_rewrite' => varchar,

     *                         'description_short' => text,

     *                         'available_later' => boolean,

     *                         'id_product' => int,

     *                         'stock_available' => int,

     *                         'out_of_stock' => int,

     *                         'id_accessory_group_product' => int

     *                         'id_product_attribute' => int,

     *                         'id_image' => int,

     *                         'name' => varchar,

     *                         'combinations' => array(

     *                         id_product_attribute => array(

     *                         'id_product_attribute' => int,

     *                         'stock_available' => int,

     *                         'out_of_stock' => int,

     *                         'id_image' => int,

     *                         'combination' => varchar,

     *                         'image' => varchar,

     *                         ),

     *                         ......................

     *                         ),

     *                         'is_available_when_out_of_stock' => int,

     *                         'is_available_for_order' => int

     *                         'is_available_buy_together' => int,

     *                         'image' => varchar

     *                         )

     *

     * @return array

     *               <pre>

     *               array(

     *               'name' => varchar,

     *               'description_short' => text,

     *               'qty' => int,

     *               'avaiable_quantity' => int,

     *               'out_of_stock' => int,

     *               'is_available_when_out_of_stock' => int,

     *               'available_later' => varchar,

     *               'id_accessory_group' => int,

     *               'id_accessory' => int,

     *               'default_id_product_attribut' => int,

     *               'default_quantity' => int,

     *               'combinations' => array(

     *               id_product_attribute => array(

     *               'id_product_attribute' => int,

     *               'stock_available' => int,

     *               'out_of_stock' => int,

     *               'id_image' => int,

     *               'combination' => varchar,

     *               'image' => varchar,

     *               ),

     *               ......................

     *               )

     *               )

     */

    protected function formatAccessory(array $accessory = array())

    {

        $default_id_product_attribute = (int) Product::getDefaultAttribute($accessory['id_accessory'], self::DEFAULT_QTY);

        $formatted_accessory = array();

        $formatted_accessory['name'] = Tools::truncate($accessory['name'], 38);

        $formatted_accessory['description_short'] = $accessory['description_short'];

        $formatted_accessory['qty'] = $accessory['default_quantity'];

        $formatted_accessory['avaiable_quantity'] = (int) $accessory['stock_available'];

        $formatted_accessory['out_of_stock'] = HsMaProduct::isAvailableWhenOutOfStock($accessory['out_of_stock']);

        $formatted_accessory['is_available_when_out_of_stock'] = (HsMaProduct::isAvailableWhenOutOfStock($accessory['out_of_stock']) && $accessory['stock_available'] < $accessory['default_quantity']) ? 1 : 0;

        $formatted_accessory['available_later'] = $this->getMessageAvailableLater($accessory['available_later']);

        $formatted_accessory['id_accessory_group'] = $accessory['id_accessory_group'];

        $formatted_accessory['id_accessory'] = $accessory['id_accessory'];

        $formatted_accessory['default_id_product_attribute'] = $accessory['id_product_attribute'] ? $accessory['id_product_attribute'] : $default_id_product_attribute;



        $valid_combinations = array($accessory['id_product_attribute'] => $accessory);

        if ($accessory['combinations']) {

            $formatted_accessory['default_quantity'] = (int) $accessory['default_quantity'] > 0 ? (int) $accessory['default_quantity'] : (int) self::DEFAULT_QTY;

            $_id_product_attributes = array();

            foreach ($accessory['combinations'] as $combination) {

                $_id_product_attributes[] = $combination['id_product_attribute'];

            }

            $valid_id_product_attributes = $accessory['id_product_attribute'] ? array($accessory['id_product_attribute']) : $_id_product_attributes;

            $valid_combinations = array_intersect_key($accessory['combinations'], array_flip($valid_id_product_attributes));

        }

        $accessory['combinations'] = $valid_combinations;

        $formatted_accessory['combinations'] = $this->formatCombinations($accessory);



        return $formatted_accessory;

    }



    /**

     * format product.

     *

     * @param Object $product Product object

     *

     * @return array

     *               <pre>

     *               array(

     *               'name' => varchar,

     *               'description_short' => text,

     *               'qty' => int,

     *               'avaiable_quantity' => int,

     *               'out_of_stock' => int,

     *               'is_available_when_out_of_stock' => int,

     *               'available_later' => varchar,

     *               'id_accessory_group' => int,

     *               'id_accessory' => int,

     *               'default_id_product_attribut' => int,

     *               'default_quantity' => int,

     *               'combinations' => array(

     *               id_product_attribute => array(

     *               'id_product_attribute' => int,

     *               'stock_available' => int,

     *               'out_of_stock' => int,

     *               'id_image' => int,

     *               'combination' => varchar,

     *               'image' => varchar,

     *               ),

     *               ......................

     *               )

     *               ) 

     */

    protected function formatProduct(Product $product)

    {

        $default_id_product_attribute = (int) Product::getDefaultAttribute($product->id, self::DEFAULT_QTY);

        $id_customer = $this->getIdCustomer();

        $use_tax = $this->isUsetax();

        $currency_decimals = $this->getDecimals();



        $formatted_product = array();

        $formatted_product['name'] = Tools::truncate($product->name, 38);

        $formatted_product['qty'] = self::DEFAULT_QTY;

        $formatted_product['out_of_stock'] = Product::isAvailableWhenOutOfStock($product->out_of_stock);

        $formatted_product['available_quantity'] = (int) $product->quantity;

        $formatted_product['description_short'] = $product->description_short;

        $formatted_product['default_id_product_attribute'] = $default_id_product_attribute;

        $formatted_product['combinations'][$default_id_product_attribute]['price'] = Tools::ps_round($product->getPrice($use_tax), $currency_decimals);

        $formatted_product['combinations'][$default_id_product_attribute]['image'] = '';

        $formatted_product['combinations'][$default_id_product_attribute]['specific_prices'] = HsMaSpecificPrice::getSpecificPrices($product->id, $id_customer, $this->getIdGroup($id_customer), $this->getIdCountry($id_customer), $this->getIdCurrency(), $this->getIdShop(), !$this->isPrestashop16());



        return $formatted_product;

    }



    /**

     * Format combinations.

     *

     * @param array $accessory

     *                         <pre>

     *                         array(

     *                         'id_accessory_group' => int,

     *                         'id_accessory' => int,

     *                         'default_quantity' => int,

     *                         'required' => boolean,

     *                         'position' => int,

     *                         'link_rewrite' => varchar,

     *                         'description_short' => text,

     *                         'available_later' => boolean,

     *                         'id_product' => int,

     *                         'stock_available' => int,

     *                         'out_of_stock' => int,

     *                         'id_accessory_group_product' => int

     *                         'id_product_attribute' => int,

     *                         'id_image' => int,

     *                         'name' => varchar,

     *                         'combinations' => array(

     *                         id_product_attribute => array(

     *                         'id_product_attribute' => int,

     *                         'stock_available' => int,

     *                         'out_of_stock' => int,

     *                         'id_image' => int,

     *                         'combination' => varchar,

     *                         'image' => varchar,

     *                         ),

     *                         ......................

     *                         ),

     *                         'is_available_when_out_of_stock' => int,

     *                         'is_available_for_order' => int

     *                         'is_available_buy_together' => int,

     *                         'image' => varchar

     *                         )

     *

     * @return array

     */

    protected function formatCombinations($accessory)

    {

        $id_customer = $this->getIdCustomer();

        $formated_combinations = array();

        foreach ($accessory['combinations'] as $id_product_attribute => $combination) {

            $product = new Product($accessory['id_accessory']);

            $formated_combinations[$id_product_attribute] = array(

                'price' => Tools::ps_round($product->getPrice($this->isUsetax(), $combination['id_product_attribute'], 2), $this->getDecimals()),

                'image_fancybox' => $this->context->link->getImageLink($accessory['link_rewrite'], $combination['id_image'], Configuration::get('HSMA_IMAGE_SIZE_IN_FANCYBOX')),

                'image_default' => $this->context->link->getImageLink($accessory['link_rewrite'], $combination['id_image'], Configuration::get('HSMA_IMAGE_TYPE')),

                'name' => !empty($combination['combination']) ? $combination['combination'] : $combination['name'],

                'specific_prices' => HsMaSpecificPrice::getSpecificPrices($accessory['id_accessory'], $id_customer, $this->getIdGroup($id_customer), $this->getIdCountry($id_customer), $this->getIdCurrency(), $this->getIdShop(), !$this->isPrestashop16(), $combination['id_product_attribute']),

                'avaiable_quantity' => (int) $combination['stock_available'],

                'out_of_stock' => HsMaProduct::isAvailableWhenOutOfStock($combination['out_of_stock']),

            );

        }



        return $formated_combinations;

    }



    /**

     * Get ID customer.

     *

     * @return int

     */

    protected function getIdCustomer()

    {

        return ($this->context->customer->isLogged()) ? (int) $this->context->customer->id : 0;

    }



    /**

     * Get ID Group.

     *

     * @param int $id_customer

     *

     * @return int

     */

    protected function getIdGroup($id_customer)

    {

        return ($id_customer ? Customer::getDefaultGroupId((int) $id_customer) : (int) Group::getCurrent()->id);

    }



    /**

     * Get ID Country.

     *

     * @param int $id_customer

     *

     * @return int

     */

    protected function getIdCountry($id_customer)

    {

        return ($id_customer ? Customer::getCurrentCountry($id_customer) : Configuration::get('PS_COUNTRY_DEFAULT'));

    }



    /**

     * Get ID shop.

     *

     * @return int

     */

    protected function getIdShop()

    {

        return (int) $this->context->shop->id;

    }



    protected function getIdCurrency()

    {

        return (int) $this->context->currency->id;

    }

    /**

     * Check if the system used tax or not.

     *

     * @return bool

     */

    protected function isUsetax()

    {

        return Product::getTaxCalculationMethod($this->context->customer->id) ? false : true;

    }



    /**

     * Get current diceimal.

     *

     * @return int

     */

    protected function getDecimals()

    {

        return (int) $this->context->currency->decimals * _PS_PRICE_DISPLAY_PRECISION_;

    }



    /**

     * Render the main block of accessories based on id_products.

     *

     * @param array $id_products

     *

     * @return html

     */

    protected function renderBlockAccessories($id_products, $is_product_page = true)

    {

        $hs_accessories_group = new HsAccessoriesGroupAbstract();

        $accessory_groups = $hs_accessories_group->getGroups((int) $this->context->language->id, true);

        // check the product has not accessories we should return

        if (empty($accessory_groups)) {

            return;

        }

        $id_groups = array();

        foreach ($accessory_groups as &$group) {

            $group['display_style'] = ((int) $group['display_style'] == (int) HsMaDisplayStyle::USE_DEFAULT) ? (int) Configuration::get('HSMA_DISPLAY_STYLE') : (int) $group['display_style'];

            $id_groups[] = $group['id_accessory_group'];

        }



        $accessories_groups = $hs_accessories_group->getAccessoriesByGroups($id_groups, $id_products, true, $this->context->language->id, true, $is_product_page);

        $accessories_table_price = array();

        $currency_decimals = $this->getDecimals();

        $use_tax = $this->isUsetax();

        $random_main_product_id = $this->getRandomId();

        if ($is_product_page) {

            $product = $this->context->controller->getProduct();

            $accessories_table_price[$random_main_product_id] = $this->formatProduct($product);

        }

        foreach ($accessories_groups as &$accessories_group) {

            foreach ($accessories_group as &$accessory) {

                $product = new Product((int) $accessory['id_accessory'], true, (int) $this->context->language->id);

                $random_product_accessories_id = $this->getRandomId();

                $default_id_product_attribute = (int) Product::getDefaultAttribute($product->id, self::DEFAULT_QTY);

                if (!Validate::isLoadedObject($product)) {

                    unset($accessory);

                    continue;

                }

                if ($is_product_page) {

                    $accessories_table_price[$random_product_accessories_id] = $this->formatAccessory($accessory);

                }

                $accessory['price'] = Tools::ps_round($product->getPrice($use_tax, (int) $accessory['id_product_attribute']), $currency_decimals);

                $accessory['random_product_accessories_id'] = $random_product_accessories_id;

                $accessory['default_id_product_attribute'] = $default_id_product_attribute;

                $accessory['link'] = $this->context->link->getProductLink($product);

                $accessory['available_later'] = $this->getMessageAvailableLater($accessory['available_later']);

            }

        }

        $this->context->smarty->assign(array(

            'accessory_configuration_keys' => Configuration::getMultiple(array_keys($this->configuration_keys)),

            'accessory_block_title' => Configuration::get('HSMA_TITLE', $this->context->language->id),

            'accessory_image_type' => Configuration::get('HSMA_IMAGE_TYPE'),

            'change_main_price' => Configuration::get('HSMA_CHANGE_MAIN_PRICE'),

            'image_size_fancybox' => Configuration::get('HSMA_IMAGE_SIZE_IN_FANCYBOX'),

            'show_table_price' => Configuration::get('HSMA_SHOW_PRICE_TABLE'),

            'accessory_groups' => $accessory_groups,

            'accessories_table_price' => Tools::jsonEncode($accessories_table_price),

            'random_main_product_id' => $random_main_product_id,

            'sub_total' => $this->i18n['sub_total'],

            'accessories_groups' => $accessories_groups,

            'static_token' => Tools::getToken(false),

            'main_product_minimal_quantity' => $is_product_page ? $product->minimal_quantity : 1,

            'buy_main_accessory_together' => HsMaProductSetting::getBuyTogetherCurrentValue($id_products[0]),

            'is_product_page' => $is_product_page,

        ));



        return $this->display($this->name.'.php', 'multi_accessories.tpl');

    }



    /**

     * Display accessories in groups within product page.

     *

     * @return HTML string

     */

    public function hookDisplayRightColumnProduct()

    {

        $id_product = Tools::getValue('id_product', false);

        if (!empty($id_product)) {

            return $this->renderBlockAccessories(array($id_product), true);

        }

    }



    /**

     * Display accessories in groups within product page.

     *

     * @return HTML string

     */

    public function hookDisplayLeftColumnProduct()

    {

        return $this->hookDisplayRightColumnProduct();

    }



    /**

     * Display accessories in groups within right column in product page.

     *

     * @return HTML string

     */

    public function hookDisplayRightColumn()

    {

        if ($this->context->controller instanceof ProductController) {

            return $this->hookDisplayRightColumnProduct();

        }

    }



    /**

     * Display accessories in groups within left column in product page.

     *

     * @return HTML string

     */

    public function hookDisplayLeftColumn()

    {

        if ($this->context->controller instanceof ProductController) {

            return $this->hookDisplayRightColumnProduct();

        }

    }



    /**

     * A custom hook so that we can place the Multi Accessories block anywhere on product page's template <br/>

     * For example:<br/>

     * {hook h="displayMultiAccessoriesProduct"}.

     *

     * @return HTML

     */

    public function hookDisplayMultiAccessoriesProduct()

    {

        return $this->hookDisplayRightColumnProduct();

    }



    /**

     * Display accessories in groups within block product button in product page.

     *

     * @return HTML string

     */

    public function hookDisplayProductButtons()

    {

        return $this->hookDisplayRightColumnProduct();

    }



    /**

     * show a tab in product tab page => We will implement this function later.

     *

     * @return HTML string

     */

    public function hookDisplayProductTab()

    {

    }



    /**

     * show all accessories in a tab of product page => We will implement this function later.

     *

     * @return HTML string

     */

    public function hookDisplayProductTabContent()

    {

    }



    /**

     * Insert acessories into shopping cart <br />

     * Update the quantity of the main product <br />.

     *

     * @return bool

     */

    public function hookActionCartSave()

    {

        if (!Validate::isLoadedObject($this->context->cart)) {

            return;

        }



        $qty = (int) Tools::getValue('qty', 1);



        $id_accesories_attributes = array();

        if (Tools::getValue('id_accesories_attributes')) {

            $id_accesories_attributes = explode(',', Tools::getValue('id_accesories_attributes'));

        }



        if (Tools::getValue('id_accessories')) {

            $this->accessories = explode(',', Tools::getValue('id_accessories'));

        }



        $custom_qty = array();

        if (Tools::getValue('custom_qty')) {

            $custom_qty = explode(',', Tools::getValue('custom_qty'));

        }



        if (self::$has_cart_been_added) {

            self::$has_cart_been_added = false;



            return;

        } else {

            if (!self::$is_executed) {

                self::$is_executed = true;

                if (empty($this->accessories)) {

                    return;

                }



                // start process insert accessories to cart

                $i = 0;

                $accessory_qty = array();

                foreach ($this->accessories as $id_accessory) {

                    $product = new Product($id_accessory, true, $this->context->language->id);

                    if (!empty($id_accesories_attributes[$i])) {

                        $id_product_attribute = (int) $id_accesories_attributes[$i];

                    } elseif ($product->hasAttributes()) {

                        $id_product_attribute = Product::getDefaultAttribute($product->id);

                    } else {

                        $id_product_attribute = 0;

                    }



                    if (empty($custom_qty)) {

                        $accessory_qty[$i] = $qty;

                    } else {

                        $accessory_qty = $custom_qty;

                    }



                    if ($product->hasAttributes()) {

                        if (!Product::isAvailableWhenOutOfStock($product->out_of_stock) && !Attribute::checkAttributeQty($id_product_attribute, $accessory_qty[$i])) {

                            $accessory_qty[$i] = Product::getQuantity($id_accessory);

                        }

                    } elseif (!$product->checkQty($accessory_qty[$i])) {

                        $accessory_qty[$i] = Product::getQuantity($id_accessory);

                    }



                    if ($accessory_qty[$i] > 0) {

                        $this->context->cart->updateQty($accessory_qty[$i], $id_accessory, $id_product_attribute);

                    }

                    ++$i;

                }

            }

        }

    }



    /**

     * Delete accessories of a product.

     */

    public function hookActionProductDelete($params)

    {

        $id_product = (int) $params['id_product'];

        if (!$id_product) {

            return;

        }

        $hs_accessories_group = new HsAccessoriesGroupAbstract();

        $hs_accessories_group->deleteAccessories($id_product, array());

    }



    /**

     * Turn on a flag in case the event (adding a main product into shopping cart) occurs.

     */

    public function hookActionObjectCartAddAfter()

    {

        self::$has_cart_been_added = true;

    }



    /**

     * get min quantity of array quantity of products.

     *

     * @param array $quantities = array (1 => int, 2 => int,....)

     *

     * @return int

     */

    protected function getMinQuantity(array $quantities)

    {

        return min($quantities);

    }



    /**

     * show a setting form in the top of list accessory group.

     *

     * @return html

     */

    public function hookDisplayAdminListBefore()

    {

        $controller_name = Tools::getValue('controller');

        if ($controller_name === $this->class_controller_admin_group) {

            $this->context->controller->addCSS($this->getCSSPath() . 'adminpage.css', 'all');

            if (!$this->isPrestashop16()) {

                $this->context->controller->addCSS($this->getCSSPath() . 'adminpage_15.css', 'all');

            }



            return $this->renderForm();

        }

    }



    /**

     * Render form settings

     * @return html

     */

    protected function renderForm()

    {

        $fields_form = array(

            'form' => array(

                'legend' => array(

                    'title' => $this->i18n['settings'],

                    'image' => $this->getImgPath() . 'setting.gif',

                ),

                'input' => array(

                    array(

                        'type' => 'select',

                        'label' => $this->i18n['display_style'],

                        'name' => 'HSMA_DISPLAY_STYLE',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['define_how_accessories_look_like_at_product_page'],

                        'options' => array(

                            'query' => $this->getDisplayStyles(),

                            'id' => 'id',

                            'name' => 'name'

                        )

                    ),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['show_images'],

                        'name' => 'HSMA_SHOW_IMAGES',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['display_images_along_with_each_accessory'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_SHOW_IMAGES_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_SHOW_IMAGES_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    $this->renderFancyBoxToImage(),

                    $this->renderImageSizeInFancyBox(),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['show_price'],

                        'name' => 'HSMA_SHOW_PRICE',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['display_prices_along_with_each_accessory'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_SHOW_PRICE_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_SHOW_PRICE_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['show_price_table'],

                        'name' => 'HSMA_SHOW_PRICE_TABLE',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['tell_your_customers_a_summary'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_SHOW_PRICE_TABLE_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_SHOW_PRICE_TABLE_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['change_the_main_item_s_price_accordingly'],

                        'name' => 'HSMA_CHANGE_MAIN_PRICE',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['whenever_an_accessory_is_added_or_removed_the_main_item_s_price_is_changed_and_your_customers_clearly_know_the_amount'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_CHANGE_MAIN_PRICE_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_CHANGE_MAIN_PRICE_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['add_custom_quantity_to_basket'],

                        'name' => 'HSMA_SHOW_CUSTOM_QUANTITY',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['allow_customer_add_custom_quantity_of_each_accessory_to_basket'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_SHOW_CUSTOM_QUANTITY_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_SHOW_CUSTOM_QUANTITY_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['buy_main_product_accessories_together'],

                        'name' => 'HSMA_BUY_ACCESSORY_MAIN_TOGETHER',

                        'form_group_class' => 'buytogetheroption',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['tell_your_customers_that_they_need_to_buy_main_product_and_accessories_together'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_BUY_ACCESSORY_MAIN_TOGETHER_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_BUY_ACCESSORY_MAIN_TOGETHER_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    $this->renderAlertMessage(),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['add_each_accessory_to_basket'],

                        'name' => 'HSMA_EACH_ACCESSORY_TO_BASKET',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['allow_customer_add_separated_accessory_to_basket'],

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_EACH_ACCESSORY_TO_BASKET_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_EACH_ACCESSORY_TO_BASKET_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    array(

                        'type' => $this->isPrestashop16() ? 'switch' : 'radio',

                        'label' => $this->i18n['open_accessories_in_a_new_tab'],

                        'name' => 'HSMA_OPEN_ACCESSORIES_IN_NEW_TAB',

                        'is_bool' => true,

                        'class' => !$this->isPrestashop16() ? 't' : '',

                        'values' => array(

                            array(

                                'id' => 'HSMA_OPEN_ACCESSORIES_IN_NEW_TAB_on',

                                'value' => 1,

                                'label' => $this->i18n['enabled']

                            ),

                            array(

                                'id' => 'HSMA_OPEN_ACCESSORIES_IN_NEW_TAB_off',

                                'value' => 0,

                                'label' => $this->i18n['disabled']

                            )

                        ),

                    ),

                    array(

                        'type' => 'text',

                        'label' => $this->i18n['title'],

                        'class' => 'field-text-input',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['title_of_accessory_block_at_product_page'],

                        'name' => 'HSMA_TITLE',

                        'lang' => true,

                    ),

                    array(

                        'type' => 'text',

                        'label' => $this->i18n['displayed_text_when_backordering_is_allowed'],

                        'class' => 'field-text-input',

                        $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['if_the_text_displayed_text_when_backordering_is_allowed_in_product_edit_page_is_empty'],

                        'name' => 'HSMA_MESSAGE_AVAILABLE_LATER',

                        'lang' => true,

                    )

                ),

                'submit' => array(

                    'title' => $this->i18n['save'],

                    'name' => 'submitSetting'

                )

            ),

        );



        if (!$this->isPrestashop16()) {

            foreach ($fields_form['form']['input'] as $key => $input)

                if (empty($input))

                    unset($fields_form['form']['input'][$key]);

        }



        $helper = new HelperForm();

        $helper->show_toolbar = false;

        $helper->table = $this->table;

        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));

        $helper->default_form_language = $lang->id;

        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;

        $this->fields_form = array();



        $helper->identifier = $this->identifier;

        $helper->submit_action = '';

        $helper->currentIndex = $this->context->link->getAdminLink($this->class_controller_admin_group, false);

        $helper->token = Tools::getAdminTokenLite($this->class_controller_admin_group);

        $helper->tpl_vars = array(

            'fields_value' => $this->getConfigurationValues(),

            'languages' => $this->context->controller->getLanguages(),

            'id_language' => $this->context->language->id

        );



        return $helper->generateForm(array($fields_form));

    }



    /**

     * Render field form alert message.

     *

     * @return array

     */

    protected function renderAlertMessage()

    {

        $fields_form = array();

        $alert_message = array(

            'type' => 'text',

            'class' => 'field-text-input',

            'form_group_class' => 'alertmessage',

            'label' => $this->i18n['alert_message'],

            $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['tell_your_customer_when_they_dont_choose_any_accessories_to_buy_together_with_main_product'],

            'name' => 'HSMA_ALERT_MESSAGE',

            'lang' => true,

        );



        // check this to make sure it works with both PS 1.5 & 1.6

        if ($this->isPrestashop16()) {

            $fields_form = $alert_message;

        } elseif (Configuration::get('HSMA_BUY_ACCESSORY_MAIN_TOGETHER')) {

            $fields_form = $alert_message;

        }



        return $fields_form;

    }



    /**

     * Render field form apply fancybox to image.

     *

     * @return array

     */

    protected function renderFancyBoxToImage()

    {

        $fields_form = array();

        $fancybox_to_image = array(

            'type' => 'switch',

            'label' => $this->i18n['apply_fancybox_to_images'],

            $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['show_accessory_images_in_a_fancybox'],

            'name' => 'HSMA_APPLY_FANCYBOX_TO_IMAGE',

            'is_bool' => true,

            'class' => !$this->isPrestashop16() ? 't' : '',

            'form_group_class' => 'apply_fancybox',

            'values' => array(

                array(

                    'id' => 'HSMA_APPLY_FANCYBOX_TO_IMAGE_on',

                    'value' => 1,

                    'label' => $this->i18n['enabled'],

                ),

                array(

                    'id' => 'HSMA_APPLY_FANCYBOX_TO_IMAGE_off',

                    'value' => 0,

                    'label' => $this->i18n['disabled'],

                ),

            ),

        );



        // check this to make sure it works with both PS 1.5 & 1.6

        if ($this->isPrestashop16()) {

            $fields_form = $fancybox_to_image;

        } elseif ((int) Configuration::get('HSMA_DISPLAY_STYLE') === 0 && (int) Configuration::get('HSMA_SHOW_IMAGES') === 1) {

            $fields_form = $fancybox_to_image;

        }



        return $fields_form;

    }



    /**

     * Render field form image size in fancybox.

     *

     * @return array

     */

    protected function renderImageSizeInFancyBox()

    {

        $get_size_images = $this->getSizeOfImages();

        $fields_form = array();

        $image_size = array(

            'type' => 'select',

            'label' => $this->i18n['image_size_in_fancybox'],

            'form_group_class' => 'image_size',

            'name' => 'HSMA_IMAGE_SIZE_IN_FANCYBOX',

            $this->isPrestashop16() ? 'hint' : 'desc' => $this->i18n['image_size_in_fancybox'],

            'options' => array(

                'query' => $get_size_images['image_sizes'],

                'id' => 'id',

                'name' => 'name',

                'default' => array(

                    'label' => $get_size_images['default']['name'] ? $get_size_images['default']['name'] : '---',

                    'value' => $get_size_images['default']['id'] ? $get_size_images['default']['id'] : 0,

                ),

            ),

        );



        // check this to make sure it works with both PS 1.5 & 1.6

        if ($this->isPrestashop16()) {

            $fields_form = $image_size;

        } elseif ((int) Configuration::get('HSMA_DISPLAY_STYLE') === 0 && (int) Configuration::get('HSMA_SHOW_IMAGES') === 1 && (int) Configuration::get('HSMA_APPLY_FANCYBOX_TO_IMAGE') === 1) {

            $fields_form = $image_size;

        }



        return $fields_form;

    }



    /**

     * Get value of configuration.

     *

     * @return array

     *               <pre>

     *               Array(

     *               'key_configuration' => int,

     *               );

     */

    protected function getConfigurationValues()

    {

        $fields_values = array(

            'HSMA_DISPLAY_STYLE' => Tools::getValue('HSMA_DISPLAY_STYLE', Configuration::get('HSMA_DISPLAY_STYLE')),

            'HSMA_SHOW_IMAGES' => Tools::getValue('HSMA_SHOW_IMAGES', Configuration::get('HSMA_SHOW_IMAGES')),

            'HSMA_SHOW_PRICE' => Tools::getValue('HSMA_SHOW_PRICE', Configuration::get('HSMA_SHOW_PRICE')),

            'HSMA_SHOW_SHORT_DESCRIPTION' => Tools::getValue('HSMA_SHOW_SHORT_DESCRIPTION', Configuration::get('HSMA_SHOW_SHORT_DESCRIPTION')),

            'HSMA_SHOW_PRICE_TABLE' => Tools::getValue('HSMA_SHOW_PRICE_TABLE', Configuration::get('HSMA_SHOW_PRICE_TABLE')),

            'HSMA_EACH_ACCESSORY_TO_BASKET' => Tools::getValue('HSMA_EACH_ACCESSORY_TO_BASKET', Configuration::get('HSMA_EACH_ACCESSORY_TO_BASKET')),

            'HSMA_OPEN_ACCESSORIES_IN_NEW_TAB' => Tools::getValue('HSMA_OPEN_ACCESSORIES_IN_NEW_TAB', Configuration::get('HSMA_OPEN_ACCESSORIES_IN_NEW_TAB')),

            'HSMA_BUY_ACCESSORY_MAIN_TOGETHER' => Tools::getValue('HSMA_BUY_ACCESSORY_MAIN_TOGETHER', Configuration::get('HSMA_BUY_ACCESSORY_MAIN_TOGETHER')),

            'HSMA_SHOW_CUSTOM_QUANTITY' => Tools::getValue('HSMA_SHOW_CUSTOM_QUANTITY', Configuration::get('HSMA_SHOW_CUSTOM_QUANTITY')),

            'HSMA_APPLY_FANCYBOX_TO_IMAGE' => Tools::getValue('HSMA_APPLY_FANCYBOX_TO_IMAGE', Configuration::get('HSMA_APPLY_FANCYBOX_TO_IMAGE')),

            'HSMA_IMAGE_SIZE_IN_FANCYBOX' => Tools::getValue('HSMA_IMAGE_SIZE_IN_FANCYBOX', Configuration::get('HSMA_IMAGE_SIZE_IN_FANCYBOX')),

            'HSMA_CHANGE_MAIN_PRICE' => Tools::getValue('HSMA_CHANGE_MAIN_PRICE', Configuration::get('HSMA_CHANGE_MAIN_PRICE')),

        );



        $languages = Language::getLanguages(false);

        foreach ($languages as $lang) {

            $fields_values['HSMA_TITLE'][$lang['id_lang']] = Tools::getValue('HSMA_TITLE', Configuration::get('HSMA_TITLE', $lang['id_lang']));

            $fields_values['HSMA_MESSAGE_AVAILABLE_LATER'][$lang['id_lang']] = Tools::getValue('HSMA_MESSAGE_AVAILABLE_LATER', Configuration::get('HSMA_MESSAGE_AVAILABLE_LATER', $lang['id_lang']));

            $fields_values['HSMA_ALERT_MESSAGE'][$lang['id_lang']] = Tools::getValue('HSMA_ALERT_MESSAGE', Configuration::get('HSMA_ALERT_MESSAGE', $lang['id_lang']));

        }



        return $fields_values;

    }



    /**

     * Get display styles.

     *

     * @param bool $use_default // use value display style of default setting

     *

     * @return array

     *               array(<pre>

     *               array(

     *               'id'	 => int,

     *               'name'	 => string,

     *               ),

     *               ...

     *               )</pre>

     */

    public function getDisplayStyles($use_default = false)

    {

        $type_options = array(

            array(

                'id' => HsMaDisplayStyle::CHECKBOX,

                'name' => $this->i18n['checkbox'], ),

            array(

                'id' => HsMaDisplayStyle::DROPDOWN,

                'name' => $this->i18n['dropdown'], ),

            array(

                'id' => HsMaDisplayStyle::RADIO,

                'name' => $this->i18n['radio'], ),

        );

        if ($use_default) {

            $type_options = array_merge(array(array('id' => HsMaDisplayStyle::USE_DEFAULT, 'name' => $this->i18n['use_default'])), $type_options);

        }



        return $type_options;

    }



    /**

     * Set default title of Multi Accessories.

     *

     * @return bool

     */

    protected function setDefaultTitle()

    {

        $languages = Language::getLanguages(false);

        $titles = array();

        if (!empty($languages)) {

            foreach ($languages as $language) {

                $titles[$language['id_lang']] = $this->i18n['must_have_accessories'];

            }

        }



        if (!empty($titles)) {

            return Configuration::updateValue('HSMA_TITLE', $titles);

        }



        return false;

    }



    /**

     * Dedicated callback to upgrading process.

     *

     * @param type $version

     *

     * @return bool

     */

    public function upgrade($version)

    {

        $flag = true;

        switch ($version) {

            case '2.0':

                $flag = (Configuration::updateValue('HSMA_DISPLAY_STYLE', 1) && Configuration::updateValue('HSMA_SHOW_IMAGES', 1) && Configuration::updateValue('HSMA_SHOW_PRICE', 1) && $this->setDefaultTitle() && $this->registerHook('displayAdminListBefore') && $this->registerHook('displayAdminProductsExtra') && $this->proccessRemoveTab($this->class_controller_admin_product_setting) && $this->installer->addNewImageType());

                break;



            case '2.1':

                $flag = (Configuration::updateValue('HSMA_EACH_ACCESSORY_TO_BASKET', 1) && Configuration::updateValue('HSMA_SHOW_PRICE_TABLE', 1));

                break;



            case '2.1.2':

                $flag = $this->registerHook('displayMultiAccessoriesProduct');

                break;



            case '2.2':

                $sql = 'UPDATE `'._DB_PREFIX_.'tab_lang` tl

						 LEFT JOIN `'._DB_PREFIX_.'tab` t ON t.`id_tab` = tl.`id_tab`

						 SET tl.`name` = "'.pSQL($this->i18n['multi_accessories']).'"

						 WHERE t.`module` = "'.pSQL($this->name).'"';



                // set default alert message when install

                $flag = (Db::getInstance()->execute($sql) && $this->setDefaultAlertMessage());

                break;



            case '2.3':

                $flag = (Configuration::updateValue('HSMA_CHANGE_MAIN_PRICE', 1) && Configuration::updateValue('HSMA_APPLY_FANCYBOX_TO_IMAGE', 1));

                break;



            case '2.4':

                $flag = (Configuration::updateValue('HSMA_OPEN_ACCESSORIES_IN_NEW_TAB', 1) && $this->installer->updateTable24());

                break;



            case '2.4.3':

                $flag = $this->installer->installTab243($this->tab_admin_welcome_page);

                break;



            case '2.4.5':

                $flag = $this->registerHook('displayFooter');

                break;



            case '2.5':

                $flag = $this->installer->updateTable25() && $this->registerHook('actionProductAdd');

                break;



            case '2.5.2':

                $flag = $this->installer->updateTable252();

                break;



            case '2.7':

                $flag = $this->installer->update27($this->tabs27);

                break;



            case '2.7.3':

                $flag = $this->installer->update273();

                break;

            case '2.8':

                $flag = $this->registerHook('displayShoppingCartFooter');

                break;

            default:

                break;

        }



        return $flag;

    }



    /**

     * create product tab "Multi Accessories" in admin product detail.

     *

     * @return html

     */

    public function hookDisplayAdminProductsExtra()

    {

        $id_product = Tools::getValue('id_product', 0);

        $product = new Product($id_product, false, $this->context->language->id);



        if (Validate::isLoadedObject($product)) {

            $this->renderFormAccessoryGroup($product);

        } else {

            $this->adminDisplayWarning($this->i18n['you_must_save_this_product_before_adding_accessories']);

        }



        $st_hsmultiaccessories = array(

            'url' => $this->urls,

            'lang' => $this->i18n,

        );

        $this->context->smarty->assign(array(

            'st_hsmultiaccessories' => Tools::jsonEncode($st_hsmultiaccessories),

            'id_product' => $id_product,

        ));



        return $this->display($this->name.'.php', 'hsma_display_admin_product_extra.tpl');

    }



    /**

     * Show  group accessories of each product.

     *

     * @param Product $product

     */

    protected function renderFormAccessoryGroup($product)

    {

        $hs_accessories_group = new HsAccessoriesGroupAbstract();

        $groups = $hs_accessories_group->getGroups((int) $this->context->language->id);

        $id_groups = array();

        $id_product = (int) $product->id;

        foreach ($groups as $group) {

            $id_groups[] = $group['id_accessory_group'];

        }

        if ($id_product && !empty($id_groups)) {

            $accessories_groups = $hs_accessories_group->getAccessoriesByGroups($id_groups, array($id_product));

        } else {

            $accessories_groups = array();

        }

        $languages = Language::getLanguages(true);

        $meta_language = array();

        foreach ($languages as $lang) {

            $meta_language[] = $lang['iso_code'];

        }

        $product_setting = new HsMaProductSetting($id_product);

        $this->context->smarty->assign(array(

            'groups' => $groups,

            'product_setting' => $product_setting,

            'is_prestashop16' => $this->isPrestashop16(),

            'default_form_language' => (int) Configuration::get('PS_LANG_DEFAULT'),

            'show_custom_quantity' => (int) Configuration::get('HSMA_SHOW_CUSTOM_QUANTITY'),

            'accessories_groups' => $accessories_groups,

            'languages' => $languages,

            'url_change_accessory_position' => $this->urls['ajaxProcessUpdateAccessoryProductPosition'],

            'img_path' => $this->getImgPath(),

            'display_styles' => $this->getDisplayStyles(true),

            'buy_together_options' => $this->getProductSettingBuyTogetherOptions(),

            'buy_together_default' => HsMaProductSetting::getBuyTogetherDefault($id_product),

        ));

    }



    /**

     * Check prestashop current version is 1.6.

     *

     * @return bool

     */

    public function isPrestashop16()

    {

        return version_compare(_PS_VERSION_, '1.6') === 1;

    }



    /**

     * Remove a tab.

     *

     * @param string $name_tab

     *

     * @return bool

     */

    protected function proccessRemoveTab($name_tab)

    {

        $flag = false;

        $id_tab = (int) Tab::getIdFromClassName((string) $name_tab);

        if ($id_tab != 0) {

            $tab = new Tab($id_tab);

            if (Validate::isLoadedObject($tab)) {

                $flag = $tab->delete();

            }

        }



        return $flag;

    }



    /**

     * Get display styles.

     *

     * @return array

     *               array

     *               (<pre>

     *               [image_sizes] => array

     *               (

     *               [0] => array

     *               (

     *               [id] => 29

     *               [name] => cart_default (80x80)

     *               )

     *               [1] => array

     *               (

     *               [id] => 32

     *               [name] => home_default (250x250)

     *               )

     *               ...

     *               )

     *               [default] => array

     *               (

     *               [id] => 34

     *               [name] => thickbox_default (800x800)

     *               )

     *               )</pre>

     */

    protected function getSizeOfImages()

    {

        $image_types = ImageType::getImagesTypes('products');

        $image_sizes = array();

        $i = 0;

        $default_image = array();

        $max_width = 0;

        $id_image_type_default = Configuration::get('HSMA_IMAGE_SIZE_IN_FANCYBOX');

        foreach ($image_types as $image_type) {

            if ($id_image_type_default > 0) {

                if ($image_type['name'] === $id_image_type_default) {

                    $default_image['id'] = $image_type['name'];

                    $default_image['name'] = $image_type['name'].' ('.$image_type['width'].'x'.$image_type['height'].')';

                } else {

                    $image_sizes[$i]['id'] = $image_type['name'];

                    $image_sizes[$i]['name'] = $image_type['name'].' ('.$image_type['width'].'x'.$image_type['height'].')';

                }

            } else {

                if ($max_width == 0) {

                    $max_width = $image_type['width'];

                    $default_image['id'] = $image_type['name'];

                    $default_image['name'] = $image_type['name'].' ('.$image_type['width'].'x'.$image_type['height'].')';

                } elseif ($max_width < $image_type['width']) {

                    $max_width = $image_type['width'];

                    $image_sizes[$i] = $default_image;

                    $default_image['id'] = $image_type['name'];

                    $default_image['name'] = $image_type['name'].' ('.$image_type['width'].'x'.$image_type['height'].')';

                } else {

                    $image_sizes[$i] = $default_image;

                }

            }

            ++$i;

        }



        return array('image_sizes' => array_map('unserialize', array_unique(array_map('serialize', $image_sizes))), 'default' => $default_image);

    }



    /**

     * Get random id.

     *

     * @return string

     */

    protected function getRandomId()

    {

        return Tools::passwdGen(8, 'NO_NUMERIC');

    }



    /**

     * Get key go to welcome page.

     *

     * @return string

     */

    public function getKeyWelcomePage()

    {

        return Tools::strtoupper(md5($this->name.$this->version));

    }



    /**

     * Duplicate accessories.

     *

     * @param array $params

     *                      <pre>

     *                      array (

     *                      [product] => Product Object

     *                      (

     *                      [tax_name] =>

     *                      [tax_rate] =>

     *                      [id_manufacturer] => 0

     *                      )

     */

    public function hookActionProductAdd($params)

    {

        $old_id_product = (int) Tools::getValue('id_product');

        if (!$old_id_product) {

            return;

        }



        return HsAccessoriesGroupProduct::duplicateAccessories($old_id_product, $params['product']->id);

    }



    /**

     * Get array values buy main product & accessory together.

     *

     * @return array

     *               array(<pre>

     *               array(

     *               'id'	 => int,

     *               'name'	 => string,

     *               ),

     *               ...

     *               )</pre>

     */

    public function getProductSettingBuyTogetherOptions()

    {

        $buy_together_options = array(

            array(

                'id' => HsMaProductSetting::BUY_TOGETHER_NO,

                'name' => $this->i18n['no'], ),

            array(

                'id' => HsMaProductSetting::BUY_TOGETHER_YES,

                'name' => $this->i18n['yes'], ),

            array(

                'id' => HsMaProductSetting::BUY_TOGETHER_USE_DEFAULT,

                'name' => $this->i18n['use_default'], ),

            array(

                'id' => HsMaProductSetting::BUY_TOGETHER_REQUIRED,

                'name' => $this->i18n['let_me_specify'], ),

        );



        return $buy_together_options;

    }



    /**

     * Get message available later of accessory.

     *

     * @param string $available_later

     *

     * @return string

     */

    protected function getMessageAvailableLater($available_later)

    {

        $message_available_later = $this->i18n['out_of_stock_but_backordering_is_allowed'];

        $config_message_available_later = Configuration::get('HSMA_MESSAGE_AVAILABLE_LATER', (int) $this->context->language->id);

        if (!empty($available_later)) {

            $message_available_later = $available_later;

        } elseif (!empty($config_message_available_later)) {

            $message_available_later = $config_message_available_later;

        }



        return $message_available_later;

    }

}

