<?php

/**

 * HsMaProduct for Multi Accessories Override 2 function of class Product: getPriceStatic and priceCalculation.

 *

 * @author    PrestaMonster.com

 * @copyright PrestaMonster.com

 * @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)

 */



class HsMaProduct extends Product

{

    /**

     * Contain array combination.

     *

     * @var array

     */

    protected static $hsma_product_combination = array();



    /**

     * Override getPriceStatic HsMaProduct::priceCalculation

     * Get product price only use for ps 1.5 get specific price from module.

     *

     * @param int  $id_product           Product id

     * @param bool $usetax               With taxes or not (optional)

     * @param int  $id_product_attribute Product attribute id (optional).

     *                                   If set to false, do not apply the combination price impact. NULL does apply the default combination price impact.

     * @param int  $decimals             Number of decimals (optional)

     * @param int  $divisor              Useful when paying many time without fees (optional)

     * @param bool $only_reduc           Returns only the reduction amount

     * @param bool $usereduc             Set if the returned amount will include reduction

     * @param int  $quantity             Required for quantity discount application (default value: 1)

     * @param bool $force_associated_tax DEPRECATED - NOT USED Force to apply the associated tax. Only works when the parameter $usetax is true

     * @param int  $id_customer          Customer ID (for customer group reduction)

     * @param int  $id_cart              Cart ID. Required when the cookie is not accessible (e.g., inside a payment module, a cron task...)

     * @param int  $id_address           Customer address ID. Required for price (tax included) calculation regarding the guest localization

     * @param variable_reference

     * @param float $specificPriceOutput.

     *                                    If a specific price applies regarding the previous parameters, this variable is filled with the corresponding SpecificPrice object

     * @param bool  $with_ecotax          insert ecotax in price output.

     *

     * @return float Product price

     */

    public static function getPriceStatic($id_product, $usetax = true, $id_product_attribute = null, $decimals = 6, $divisor = null, $only_reduc = false, $usereduc = true, $quantity = 1, $force_associated_tax = false, $id_customer = null, $id_cart = null, $id_address = null, &$specific_price_output = null, $with_ecotax = true, $use_group_reduction = true, Context $context = null, $use_customer_price = true)

    {

        if (!$context) {

            $context = Context::getContext();

        }



        $cur_cart = $context->cart;



        if ($divisor !== null) {

            Tools::displayParameterAsDeprecated('divisor');

        }



        if (!Validate::isBool($usetax) || !Validate::isUnsignedId($id_product)) {

            die(Tools::displayError());

        }



        // Initializations

        $id_group = (int) Group::getCurrent()->id;



        // If there is cart in context or if the specified id_cart is different from the context cart id

        if (!is_object($cur_cart) || (Validate::isUnsignedInt($id_cart) && $id_cart && $cur_cart->id != $id_cart)) {

            /*

            * When a user (e.g., guest, customer, Google...) is on PrestaShop, he has already its cart as the global (see /init.php)

            * When a non-user calls directly this method (e.g., payment module...) is on PrestaShop, he does not have already it BUT knows the cart ID

            * When called from the back office, cart ID can be inexistant

            */

            if (!$id_cart && !isset($context->employee)) {

                die(Tools::displayError());

            }

            $cur_cart = new Cart($id_cart);

            // Store cart in context to avoid multiple instantiations in BO

            if (!Validate::isLoadedObject($context->cart)) {

                $context->cart = $cur_cart;

            }

        }



        $cart_quantity = 0;

        if ((int) $id_cart) {

            $cache_id = 'HsMaProduct::getPriceStatic_'.(int) $id_product.'-'.(int) $id_cart;

            if (!Cache::isStored($cache_id) || ($cart_quantity = Cache::retrieve($cache_id) != (int) $quantity)) {

                $sql = 'SELECT SUM(`quantity`)

				FROM `'._DB_PREFIX_.'cart_product`

				WHERE `id_product` = '.(int) $id_product.'

				AND `id_cart` = '.(int) $id_cart;

                $cart_quantity = (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($sql);

                Cache::store($cache_id, $cart_quantity);

            }

            $cart_quantity = Cache::retrieve($cache_id);

        }



        $id_currency = (int) Validate::isLoadedObject($context->currency) ? $context->currency->id : Configuration::get('PS_CURRENCY_DEFAULT');



        // retrieve address informations

        $id_country = (int) $context->country->id;

        $id_state = 0;

        $zipcode = 0;



        if (!$id_address && Validate::isLoadedObject($cur_cart)) {

            $id_address = $cur_cart->{Configuration::get('PS_TAX_ADDRESS_TYPE')};

        }



        if ($id_address) {

            $address_infos = Address::getCountryAndState($id_address);

            if ($address_infos['id_country']) {

                $id_country = (int) $address_infos['id_country'];

                $id_state = (int) $address_infos['id_state'];

                $zipcode = $address_infos['postcode'];

            }

        } elseif (isset($context->customer->geoloc_id_country)) {

            $id_country = (int) $context->customer->geoloc_id_country;

            $id_state = (int) $context->customer->id_state;

            $zipcode = (int) $context->customer->postcode;

        }



        if (Tax::excludeTaxeOption()) {

            $usetax = false;

        }



        if ($usetax != false

            && !empty($address_infos['vat_number'])

            && $address_infos['id_country'] != Configuration::get('VATNUMBER_COUNTRY')

            && Configuration::get('VATNUMBER_MANAGEMENT')) {

            $usetax = false;

        }



        if (is_null($id_customer) && Validate::isLoadedObject($context->customer)) {

            $id_customer = $context->customer->id;

        }



        return self::priceCalculation(

            $context->shop->id,

            $id_product,

            $id_product_attribute,

            $id_country,

            $id_state,

            $zipcode,

            $id_currency,

            $id_group,

            $cart_quantity,

            $usetax,

            $decimals,

            $only_reduc,

            $usereduc,

            $with_ecotax,

            $specific_price_output,

            $use_group_reduction,

            $id_customer,

            $use_customer_price,

            $id_cart,

            $quantity

        );

    }



    /**

     * Override get specific price from module HsMaSpecificPrice::getSpecificPrice

     * Price calculation / Get product price.

     *

     * @param int                $id_shop               Shop id

     * @param int                $id_product            Product id

     * @param int                $id_product_attribute  Product attribute id

     * @param int                $id_country            Country id

     * @param int                $id_state              State id

     * @param int                $id_currency           Currency id

     * @param int                $id_group              Group id

     * @param int                $quantity              Quantity Required for Specific prices : quantity discount application

     * @param bool               $use_tax               with (1) or without (0) tax

     * @param int                $decimals              Number of decimals returned

     * @param bool               $only_reduc            Returns only the reduction amount

     * @param bool               $use_reduc             Set if the returned amount will include reduction

     * @param bool               $with_ecotax           insert ecotax in price output.

     * @param variable_reference $specific_price_output

     *                                                  If a specific price applies regarding the previous parameters, this variable is filled with the corresponding SpecificPrice object

     *

     * @return float Product price

     **/

    public static function priceCalculation($id_shop, $id_product, $id_product_attribute, $id_country, $id_state, $zipcode, $id_currency, $id_group, $quantity, $use_tax, $decimals, $only_reduc, $use_reduc, $with_ecotax, &$specific_price, $use_group_reduction, $id_customer = 0, $use_customer_price = true, $id_cart = 0, $real_quantity = 0)

    {

        static $address = null;

        static $context = null;



        if ($address === null) {

            $address = new Address();

        }



        if ($context == null) {

            $context = Context::getContext()->cloneContext();

        }



        if ($id_shop !== null && $context->shop->id != (int) $id_shop) {

            $context->shop = new Shop((int) $id_shop);

        }



        if (!$use_customer_price) {

            $id_customer = 0;

        }



        if ($id_product_attribute === null) {

            $id_product_attribute = Product::getDefaultAttribute($id_product);

        }



        $cache_id = $id_product.'-'.$id_shop.'-'.$id_currency.'-'.$id_country.'-'.$id_state.'-'.$zipcode.'-'.$id_group.

            '-'.$quantity.'-'.$id_product_attribute.'-'.($use_tax ? '1' : '0').'-'.$decimals.'-'.($only_reduc ? '1' : '0').

            '-'.($use_reduc ? '1' : '0').'-'.$with_ecotax.'-'.$id_customer.'-'.(int) $use_group_reduction.'-'.(int) $id_cart.'-'.(int) $real_quantity;



        // reference parameter is filled before any returns

        $specific_price = HsMaSpecificPrice::getSpecificPrice(

            (int) $id_product,

            $id_shop,

            $id_currency,

            $id_country,

            $id_group,

            $quantity,

            $id_product_attribute,

            $id_customer,

            $id_cart,

            $real_quantity

        );



        if (isset(self::$_prices[$cache_id])) {

            return self::$_prices[$cache_id];

        }



        // fetch price & attribute price

        $cache_id_2 = $id_product.'-'.$id_shop;

        if (!isset(self::$_pricesLevel2[$cache_id_2])) {

            $sql = new DbQuery();

            $sql->select('product_shop.`price`, product_shop.`ecotax`');

            $sql->from('product', 'p');

            $sql->innerJoin('product_shop', 'product_shop', '(product_shop.id_product=p.id_product AND product_shop.id_shop = '.(int) $id_shop.')');

            $sql->where('p.`id_product` = '.(int) $id_product);

            if (Combination::isFeatureActive()) {

                $sql->select('product_attribute_shop.id_product_attribute, product_attribute_shop.`price` AS attribute_price, product_attribute_shop.default_on');

                $sql->leftJoin('product_attribute', 'pa', 'pa.`id_product` = p.`id_product`');

                $sql->leftJoin('product_attribute_shop', 'product_attribute_shop', '(product_attribute_shop.id_product_attribute = pa.id_product_attribute AND product_attribute_shop.id_shop = '.(int) $id_shop.')');

            } else {

                $sql->select('0 as id_product_attribute');

            }



            $res = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);



            foreach ($res as $row) {

                $array_tmp = array(

                    'price' => $row['price'],

                    'ecotax' => $row['ecotax'],

                    'attribute_price' => (isset($row['attribute_price']) ? $row['attribute_price'] : null),

                );

                self::$_pricesLevel2[$cache_id_2][(int) $row['id_product_attribute']] = $array_tmp;



                if (isset($row['default_on']) && $row['default_on'] == 1) {

                    self::$_pricesLevel2[$cache_id_2][0] = $array_tmp;

                }

            }

        }

        if (!isset(self::$_pricesLevel2[$cache_id_2][(int) $id_product_attribute])) {

            return;

        }



        $result = self::$_pricesLevel2[$cache_id_2][(int) $id_product_attribute];



        if (!$specific_price || $specific_price['price'] < 0) {

            $price = (float) $result['price'];

        } else {

            $price = (float) $specific_price['price'];

        }

        // convert only if the specific price is in the default currency (id_currency = 0)

        if (!$specific_price || !($specific_price['price'] >= 0 && $specific_price['id_currency'])) {

            $price = Tools::convertPrice($price, $id_currency);

        }



        // Attribute price

        if (is_array($result) && (!$specific_price || !$specific_price['id_product_attribute'] || $specific_price['price'] < 0)) {

            $attribute_price = Tools::convertPrice($result['attribute_price'] !== null ? (float) $result['attribute_price'] : 0, $id_currency);

            // If you want the default combination, please use NULL value instead

            if ($id_product_attribute !== false) {

                $price += $attribute_price;

            }

        }



        // Tax

        $address->id_country = $id_country;

        $address->id_state = $id_state;

        $address->postcode = $zipcode;



        $tax_manager = TaxManagerFactory::getManager($address, Product::getIdTaxRulesGroupByIdProduct((int) $id_product, $context));

        $product_tax_calculator = $tax_manager->getTaxCalculator();



        // Add Tax

        if ($use_tax) {

            $price = $product_tax_calculator->addTaxes($price);

        }



        // Reduction

        $reduc = 0;

        if (($only_reduc || $use_reduc) && $specific_price) {

            if ($specific_price['reduction_type'] == 'amount') {

                $reduction_amount = $specific_price['reduction'];



                if (!$specific_price['id_currency']) {

                    $reduction_amount = Tools::convertPrice($reduction_amount, $id_currency);

                }

                $reduc = !$use_tax ? $product_tax_calculator->removeTaxes($reduction_amount) : $reduction_amount;

            } else {

                $reduc = $price * $specific_price['reduction'];

            }

        }



        if ($only_reduc) {

            return Tools::ps_round($reduc, $decimals);

        }

        if ($use_reduc) {

            $price -= $reduc;

        }



        // Group reduction

        if ($use_group_reduction) {

            $reduction_from_category = GroupReduction::getValueForProduct($id_product, $id_group);

            if ($reduction_from_category !== false) {

                $price -= $price * (float) $reduction_from_category;

            } else { // apply group reduction if there is no group reduction for this category

                $price *= ((100 - Group::getReductionByIdGroup($id_group)) / 100);

            }

        }



        // Eco Tax

        if (($result['ecotax'] || isset($result['attribute_ecotax'])) && $with_ecotax) {

            $ecotax = $result['ecotax'];

            if (isset($result['attribute_ecotax']) && $result['attribute_ecotax'] > 0) {

                $ecotax = $result['attribute_ecotax'];

            }



            if ($id_currency) {

                $ecotax = Tools::convertPrice($ecotax, $id_currency);

            }

            if ($use_tax) {

                // reinit the tax manager for ecotax handling

                $tax_manager = TaxManagerFactory::getManager(

                    $address,

                    (int) Configuration::get('PS_ECOTAX_TAX_RULES_GROUP_ID')

                );

                $ecotax_tax_calculator = $tax_manager->getTaxCalculator();

                $price += $ecotax_tax_calculator->addTaxes($ecotax);

            } else {

                $price += $ecotax;

            }

        }

        $price = Tools::ps_round($price, $decimals);

        if ($price < 0) {

            $price = 0;

        }



        self::$_prices[$cache_id] = $price;



        return self::$_prices[$cache_id];

    }



    /**

     * Get all combinations of product.

     *

     * @param int  $id_product

     * @param int  $id_shop

     * @param int  $id_lang

     * @param bool $hiden_attribute_name

     *

     * @return array

     *               <pre/>

     *               array (

     *               [id_product_attribute] => array (

     *               [id_product_attribute] => int

     *               [quantity] => int

     *               [combination] => string

     *               )

     *               [id_product_attribute] => array ()

     *               )

     */

    public static function getCombinations($id_product, $id_shop, $id_lang = null, $hiden_attribute_name = false)

    {

        $context = Context::getcontext();



        if (is_null($id_lang)) {

            $id_lang = (int) $context->language->id;

        }



        $id_cache = $id_product.'-'.$id_shop.'-'.$id_lang;



        if (isset(self::$hsma_product_combination[$id_cache]) && self::$hsma_product_combination[$id_cache] !== null) {

            return self::$hsma_product_combination[$id_cache];

        }



        $sql = '

				SELECT

					pa.`id_product_attribute`,

					stock.`quantity` `stock_available`,

					stock.`out_of_stock`,

					pai.`id_image`,';



        if ($hiden_attribute_name) {

            $sql .= ' GROUP_CONCAT( DISTINCT CONCAT(agl.`public_name`, " : " , al.`name`) ORDER BY agl.`public_name` SEPARATOR " - ") AS `combination` ';

        } else {

            $sql .= ' GROUP_CONCAT( DISTINCT CONCAT( al.`name`) ORDER BY agl.`public_name`, al.`name` SEPARATOR " - ") AS `combination` ';

        }



        $sql .=    '

				FROM

					`'._DB_PREFIX_.'product` p



				INNER JOIN

					`'._DB_PREFIX_.'product_shop` ps

				ON

					ps.`id_product` = p.`id_product`

				AND

					ps.`id_shop` = '.(int) $id_shop.'



				LEFT JOIN

					`'._DB_PREFIX_.'product_attribute` pa

				ON

					pa.`id_product` = p.`id_product`



				LEFT JOIN

					`'._DB_PREFIX_.'product_attribute_shop` pas

				ON

					pas.`id_product_attribute` = pa.`id_product_attribute`

					AND pas.`id_shop` = '.(int) $id_shop.'



				LEFT JOIN

					`'._DB_PREFIX_.'product_attribute_combination` pac

				ON

					pac.`id_product_attribute` = pa.`id_product_attribute`



				LEFT JOIN

					`'._DB_PREFIX_.'attribute` a

				ON

					a.`id_attribute` = pac.`id_attribute`



				LEFT JOIN

					`'._DB_PREFIX_.'attribute_group` ag

				ON

					ag.`id_attribute_group` = a.`id_attribute_group`



				LEFT JOIN

					`'._DB_PREFIX_.'attribute_lang` al

				ON

					a.`id_attribute` = al.`id_attribute`

					AND al.`id_lang` = '.(int) $id_lang.'



				LEFT JOIN

					`'._DB_PREFIX_.'attribute_group_lang` agl

				ON

					ag.`id_attribute_group` = agl.`id_attribute_group`

					AND agl.`id_lang` = '.(int) $id_lang.'



				LEFT JOIN

					`'._DB_PREFIX_.'attribute_shop` ats

				ON

					ats.`id_attribute` = a.`id_attribute`

				AND

					ats.`id_shop` = '.(int) $id_shop.'



				LEFT JOIN

					`'._DB_PREFIX_.'product_attribute_image` pai

				ON

					pai.`id_product_attribute` = pa.`id_product_attribute`



				'.Product::sqlStock('p', 'pa', false, $context->shop).'



				WHERE

					p.`id_product` = '.(int) $id_product.'

					AND p.`active` = 1



				GROUP BY

					p.`id_product`, pa.`id_product_attribute` ASC



				ORDER BY  `combination`	';

        $combinations = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);

        $formatted_combinations = array();

        foreach ($combinations as $combination) {

            if (!empty($combination['id_product_attribute'])) {

                $formatted_combinations[$combination['id_product_attribute']] = $combination;

            }

        }

        self::$hsma_product_combination[$id_cache] = $formatted_combinations;



        return self::$hsma_product_combination[$id_cache];

    }



    /**

     * Get stock status of product.

     *

     * @param int  $id_product

     * @param int  $id_product_attribute

     * @param Shop $shop

     *

     * @return array

     *               array(<pre>

     *               'out_of_stock' => int,

     *               'quantity' => int

     *               );</pre>

     */

    public static function getStockStatus($id_product, $id_product_attribute, $shop)

    {

        $sql = 'SELECT stock.`out_of_stock`, IFNULL(stock.`quantity`, 0) as `quantity`

				FROM `'._DB_PREFIX_.'product` p

				'.Product::sqlStock('p', (int) $id_product_attribute, true, $shop).'

				WHERE p.`id_product` = '.(int) $id_product;



        return Db::getInstance()->getRow($sql);

    }

}

